/*  -*- pse-c -*-
 *----------------------------------------------------------------------------
 * Filename: iegd_interface.c
 * $Revision: 1.1.2.6 $
 *----------------------------------------------------------------------------
 * Gart and DRM driver for Intel Embedded Graphics Driver
 * Copyright © 2007, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 
 * 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 *
 *----------------------------------------------------------------------------
 */

#include "global.h"
#include "intelpci.h"

/* will point to the current table entries for
 * current chipset */
gart_dispatch_t *gart_id;

/* Private data that contained chipset information */
dev_private_data_t private_data;

int iegd_find_device(u16 device)
{
	struct pci_dev *device_pdev;

	device_pdev = pci_get_device(PCI_VENDOR_ID_INTEL, device, NULL);
	/* Check for function 0. */
	if(device_pdev && PCI_FUNC(device_pdev->devfn) != 0) {
		device_pdev = pci_get_device(PCI_VENDOR_ID_INTEL,
			device, device_pdev);
	}

	if(!device_pdev) {
		return 0;
	}

	private_data.pdev = device_pdev;
	return 1;

}

/**
 * This function is to hook the function pointer that
 * belong to specific chipset, other than that this
 * is the place for customization of the structure
 * in case chipset in the same family have different
 * architecture. Make sure to add new device id here
 * if new device been introduce.
 *
 * parameter:
 *	driver_hook - Pointer to hold the structure
 *	did  - device id
 *	list - lookup table for the chipset family
 *
 * return value:
 *	0 - success
 *	1 - No function hook
 */
int bridge_driver_init(bridge_driver_t **driver_hook,
	unsigned short did, dispatch_table_t *list )
{

	(*driver_hook) = (bridge_driver_t *)dispatch_acquire(
		gart_id->device_id, list);

	/* For specific chipset implementation assign the pointer
	 * here. */
	switch(did) {
		case PCI_DEVICE_ID_Q35:
			(*driver_hook)->aperture_sizes = iegd_iq35_sizes;
			(*driver_hook)->num_aperture_sizes = 3;
			break;
		case PCI_DEVICE_ID_915GD:
		case PCI_DEVICE_ID_915AL:
		case PCI_DEVICE_ID_945G:
		case PCI_DEVICE_ID_945GM:
		case PCI_DEVICE_ID_945GME:
			(*driver_hook)->aperture_sizes = iegd_i915_sizes;
			(*driver_hook)->num_aperture_sizes = 4;
			break;
		case PCI_DEVICE_ID_965G:
		case PCI_DEVICE_ID_946GZ:
		case PCI_DEVICE_ID_G965:
		case PCI_DEVICE_ID_Q965:
		case PCI_DEVICE_ID_GM965:
		case PCI_DEVICE_ID_GME965:
			(*driver_hook)->aperture_sizes = iegd_i965_sizes;
			(*driver_hook)->num_aperture_sizes = 4;
			break;
		case PCI_DEVICE_ID_810:
		case PCI_DEVICE_ID_810DC:
		case PCI_DEVICE_ID_810E:
		case PCI_DEVICE_ID_815:
			(*driver_hook)->aperture_sizes = intel_i810_sizes;
			(*driver_hook)->num_aperture_sizes = 2;
			(*driver_hook)->create_gatt_table = agp_generic_create_gatt_table;
			(*driver_hook)->free_gatt_table   = agp_generic_free_gatt_table;
			break;
		case PCI_DEVICE_ID_830M:
		case PCI_DEVICE_ID_845G:
		case PCI_DEVICE_ID_855:
		case PCI_DEVICE_ID_865G:
			(*driver_hook)->aperture_sizes = intel_i830_sizes;
			(*driver_hook)->num_aperture_sizes = 4;
			(*driver_hook)->create_gatt_table = iegd_alm_create_gatt_table;
			(*driver_hook)->free_gatt_table   = iegd_cmn_free_gatt_table;
			break;
		default:
			return -1;
	}

	return 0;

}
